import os
import pathlib
import sys

REPO_ROOT = pathlib.Path(__file__).parent.parent.parent
sys.path.append(str(REPO_ROOT))

# This script should be careful not to disrupt automatic mechanism which
# may be expecting certain stdout, always print to stderr.
sys.stdout = sys.stderr

from bazel.wrapper_hook.install_modules import install_modules
from bazel.wrapper_hook.wrapper_debug import wrapper_debug

wrapper_debug(f"wrapper hook script is using {sys.executable}")


def main():
    install_modules(sys.argv[1])

    from bazel.wrapper_hook.autogenerated_targets import autogenerate_targets
    from bazel.wrapper_hook.engflow_check import engflow_auth
    from bazel.wrapper_hook.generate_common_bes_bazelrc import write_workstation_bazelrc
    from bazel.wrapper_hook.plus_interface import check_bazel_command_type, test_runner_interface
    from bazel.wrapper_hook.remove_auto_header_dirs import clean_up_auto_header_dirs
    from bazel.wrapper_hook.set_mongo_variables import write_mongo_variables_bazelrc

    clean_up_auto_header_dirs(REPO_ROOT)

    # This is used to autogenerate a BUILD.bazel that creates
    # Filegroups for select tags - used to group targets for installing
    autogenerate_targets(sys.argv, sys.argv[1])

    enterprise = True
    if check_bazel_command_type(sys.argv[1:]) not in ["clean", "shutdown", "version", None]:
        args = sys.argv
        enterprise_mod = REPO_ROOT / "src" / "mongo" / "db" / "modules" / "enterprise"
        if not enterprise_mod.exists():
            enterprise = False
            print(
                f"{enterprise_mod.relative_to(REPO_ROOT).as_posix()} missing, defaulting to local non-enterprise build (--config=local --build_enterprise=False). Add the directory to not automatically add these options."
            )
            args += ["--build_enterprise=False", "--config=local"]

        if any(arg.startswith("--include_mongot") for arg in args):
            os.makedirs("mongot-localdev", exist_ok=True)

        engflow_auth(args)

        write_workstation_bazelrc(args)

        write_mongo_variables_bazelrc(args)

        args = test_runner_interface(
            sys.argv[1:],
            autocomplete_query=os.environ.get("MONGO_AUTOCOMPLETE_QUERY") == "1",
            enterprise=enterprise,
        )

    else:
        args = sys.argv[2:]

    os.chmod(os.environ.get("MONGO_BAZEL_WRAPPER_ARGS"), 0o644)
    with open(os.environ.get("MONGO_BAZEL_WRAPPER_ARGS"), "w") as f:
        f.write("\n".join(args))
        f.write("\n")


if __name__ == "__main__":
    main()
