# Command Dispatch

Command dispatch refers to the general process by which client requests are
taken from the network, parsed, sanitized, then finally run on databases.

## Service Entry Points

[Service entry points][service_entry_point_h] fulfill the transition from the
transport layer into command implementations. For each incoming connection
from a client (in the form of a [session][session_h] object), a new dedicated
thread is spawned then detached, and is also assigned a new [session workflow]
[session_workflow_h], responsible for maintaining the workflow of a
single client connection during its lifetime. Central to the entry point is the
`handleRequest()` function, which manages the server-side logic of processing
requests and returns a response message indicating the result of the
corresponding request message. This function is currently implemented by several
subclasses of the parent `ServiceEntryPoint` in order to account for the
differences in processing requests between the shard and router roles -- these
distinctions are reflected in the `ServiceEntryPointRouterRole` and
`ServiceEntryPointShardRole` subclasses (see [here][service_entry_point_router_role_h]
and [here][service_entry_point_shard_role.h]).

## Strategy

One area in which the _mongos_ entry point differs from its _mongod_ counterpart
is in its usage of the [Strategy class][strategy_h]. `Strategy` operates as a
legacy interface for processing client read, write, and command requests; there
is a near 1-to-1 mapping between its constituent functions and request types
(e.g. `writeOp()` for handling write operation requests, `getMore()` for a
getMore request, etc.). These functions comprise the backbone of the _mongos_
entry point's `handleRequest()` -- that is to say, when a valid request is
received, it is sieved and ultimately passed along to the appropriate Strategy
class member function. The significance of using the Strategy class specifically
with the _mongos_ entry point is that it [facilitates query routing to
shards][mongos_router] in _addition_ to running queries against targeted
databases (see [s/transaction_router.h][transaction_router_h] for finer
details).

## Commands

The [Command class][commands_h] serves as a means of cataloging a server command
as well as ascribing various attributes and behaviors to commands via the [type
system][template_method_pattern], that will likely be used during the lifespan
of a particular server. Construction of a Command should only occur during
server startup. When a new Command is constructed, that Command is stored in a
global `CommandRegistry` object for future reference. There are two kinds of
Command subclasses: `BasicCommand` and `TypedCommand`.

A major distinction between the two is in their implementation of the `parse()`
member function. `parse()` takes in a request and returns a handle to a single
invocation of a particular Command (represented by a `CommandInvocation`), that
can then be used to run the Command. The `BasicCommand::parse()` is a naive
implementation that merely forwards incoming requests to the Invocation and
makes sure that the Command does not support document sequences. The
implementation of `TypedCommand::parse()`, on the other hand, varies depending
on the Request type parameter the Command takes in. Since the `TypedCommand`
accepts requests generated by IDL, the parsing function associated with a usable
Request type must allow it to be parsed as an IDL command. In handling requests,
both the _mongos_ and _mongod_ entry points interact with the Command subclasses
through the `CommandHelpers` struct in order to parse requests and ultimately
run them as Commands.

## Admission control

To ensure stability of our servers, we have implemented different admission control mechanisms to prevent data-nodes from becoming overloaded with operations. When implementing a new command, it's important to decide whether the command will be subject to one of the admission controls in place and understand the resulting outcomes.

For example, user commands may be subject to Ingress Admission Control, which happens in the [ServiceEntryPoint][IngressControl].
For information on admission control and how to implement admission control into a new command, please see [Admission Control README][ACReadMe]

## See Also

For details on transport internals, including ingress networking, see [this document][transport_internals].

[service_entry_point_h]: ../src/mongo/transport/service_entry_point.h
[session_h]: ../src/mongo/transport/session.h
[session_workflow_h]: ../src/mongo/transport/session_workflow.h
[service_entry_point_router_role_h]: ../src/mongo/s/service_entry_point_router_role.h
[service_entry_point_shard_role_h]: ../src/mongo/db/service_entry_point_shard_role.h
[read_concern]: https://docs.mongodb.com/manual/reference/read-concern/
[write_concern]: https://docs.mongodb.com/manual/reference/write-concern/
[strategy_h]: ../src/mongo/s/commands/strategy.h
[mongos_router]: https://docs.mongodb.com/manual/core/sharded-cluster-query-router/
[transaction_router_h]: ../src/mongo/s/transaction_router.h
[commands_h]: ../src/mongo/db/commands.h
[template_method_pattern]: https://en.wikipedia.org/wiki/Template_method_pattern
[transport_internals]: ../src/mongo/transport/README.md
[ACReadMe]: ../src/mongo/db/admission/README.md
[IngressControl]: https://github.com/mongodb/mongo/blob/a86c7f5de2a5de4d2f49e40e8970754ec6a5ba6c/src/mongo/db/service_entry_point_shard_role.cpp#L1803
