/**
 *    Copyright (C) 2018-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#include "mongo/db/pipeline/document_source_sort.h"

#include "mongo/base/error_codes.h"
#include "mongo/bson/bsonelement.h"
#include "mongo/bson/bsonmisc.h"
#include "mongo/bson/bsonobj.h"
#include "mongo/bson/bsonobjbuilder.h"
#include "mongo/bson/json.h"
#include "mongo/db/exec/agg/document_source_to_stage_registry.h"
#include "mongo/db/exec/document_value/document_value_test_util.h"
#include "mongo/db/pipeline/aggregation_context_fixture.h"
#include "mongo/db/pipeline/dependencies.h"
#include "mongo/db/pipeline/document_source_limit.h"
#include "mongo/db/pipeline/document_source_mock.h"
#include "mongo/db/pipeline/document_source_project.h"
#include "mongo/db/pipeline/document_source_single_document_transformation.h"
#include "mongo/db/pipeline/expression_context_for_test.h"
#include "mongo/db/pipeline/pipeline.h"
#include "mongo/db/query/explain_options.h"
#include "mongo/db/query/query_shape/serialization_options.h"
#include "mongo/idl/server_parameter_test_util.h"
#include "mongo/unittest/death_test.h"
#include "mongo/unittest/temp_dir.h"
#include "mongo/unittest/unittest.h"
#include "mongo/util/intrusive_counter.h"
#include "mongo/util/str.h"

#include <cstddef>
#include <deque>
#include <iterator>
#include <list>
#include <string>
#include <vector>

#include <boost/move/utility_core.hpp>
#include <boost/optional/optional.hpp>
#include <boost/smart_ptr/intrusive_ptr.hpp>

namespace mongo {

namespace {

using boost::intrusive_ptr;
using std::deque;
using std::string;
using std::vector;

static const BSONObj metaTextScore = BSON("$meta" << "textScore");

class DocumentSourceSortTest : public AggregationContextFixture {
protected:
    void createSort(const BSONObj& sortKey = BSON("a" << 1)) {
        BSONObj spec = BSON("$sort" << sortKey);
        BSONElement specElement = spec.firstElement();
        _sort = DocumentSourceSort::createFromBson(specElement, getExpCtx());
        checkBsonRepresentation(spec);
    }

    void setSort(intrusive_ptr<DocumentSource> sort) {
        _sort = std::move(sort);
    }

    /** Assert that iterator state accessors consistently report the source is exhausted. */
    void assertEOF() const {
        auto stage = exec::agg::buildStage(_sort);
        ASSERT(stage->getNext().isEOF());
        ASSERT(stage->getNext().isEOF());
        ASSERT(stage->getNext().isEOF());
    }

    DocumentSourceSort* sort() {
        return dynamic_cast<DocumentSourceSort*>(_sort.get());
    }

private:
    /**
     * Check that the BSON representation generated by the souce matches the BSON it was
     * created with.
     */
    void checkBsonRepresentation(const BSONObj& spec) {
        vector<Value> arr;
        _sort->serializeToArray(arr);
        BSONObj generatedSpec = arr[0].getDocument().toBson();
        ASSERT_BSONOBJ_EQ(spec, generatedSpec);
    }
    intrusive_ptr<DocumentSource> _sort;
};


TEST_F(DocumentSourceSortTest, RejectsNonObjectSpec) {
    BSONObj spec = BSON("$sort" << 1);
    BSONElement specElement = spec.firstElement();
    ASSERT_THROWS(DocumentSourceSort::createFromBson(specElement, getExpCtx()), AssertionException);
}

TEST_F(DocumentSourceSortTest, RejectsEmptyObjectSpec) {
    BSONObj spec = BSON("$sort" << BSONObj());
    BSONElement specElement = spec.firstElement();
    ASSERT_THROWS(DocumentSourceSort::createFromBson(specElement, getExpCtx()), AssertionException);
}

TEST_F(DocumentSourceSortTest, RejectsSpecWithNonNumericValues) {
    BSONObj spec = BSON("$sort" << BSON("a" << "b"));
    BSONElement specElement = spec.firstElement();
    ASSERT_THROWS(DocumentSourceSort::createFromBson(specElement, getExpCtx()), AssertionException);
}

TEST_F(DocumentSourceSortTest, RejectsSpecWithZeroAsValue) {
    BSONObj spec = BSON("$sort" << BSON("a" << 0));
    BSONElement specElement = spec.firstElement();
    ASSERT_THROWS(DocumentSourceSort::createFromBson(specElement, getExpCtx()), AssertionException);
}

TEST_F(DocumentSourceSortTest, SortWithLimit) {
    auto expCtx = getExpCtx();
    createSort(BSON("a" << 1));

    ASSERT(!sort()->getLimit());
    DocumentSourceContainer container;
    container.push_back(sort());

    {  // pre-limit checks
        vector<Value> arr;
        sort()->serializeToArray(arr);
        ASSERT_BSONOBJ_EQ(arr[0].getDocument().toBson(), BSON("$sort" << BSON("a" << 1)));

        ASSERT(sort()->distributedPlanLogic());
        ASSERT(sort()->distributedPlanLogic()->shardsStage != nullptr);
        ASSERT(sort()->distributedPlanLogic()->mergingStages.empty());
    }

    container.push_back(DocumentSourceLimit::create(expCtx, 10));
    sort()->optimizeAt(container.begin(), &container);
    ASSERT_EQUALS(container.size(), 1U);
    ASSERT_EQUALS(*sort()->getLimit(), 10);

    // unchanged
    container.push_back(DocumentSourceLimit::create(expCtx, 15));
    sort()->optimizeAt(container.begin(), &container);
    ASSERT_EQUALS(container.size(), 1U);
    ASSERT_EQUALS(*sort()->getLimit(), 10);

    // reduced
    container.push_back(DocumentSourceLimit::create(expCtx, 5));
    sort()->optimizeAt(container.begin(), &container);
    ASSERT_EQUALS(container.size(), 1U);
    ASSERT_EQUALS(*sort()->getLimit(), 5);

    vector<Value> arr;
    sort()->serializeToArray(arr);
    ASSERT_VALUE_EQ(Value(arr), DOC_ARRAY(DOC("$sort" << DOC("a" << 1)) << DOC("$limit" << 5)));

    ASSERT(sort()->distributedPlanLogic());
    ASSERT(sort()->distributedPlanLogic()->shardsStage != nullptr);
    ASSERT_EQ(sort()->distributedPlanLogic()->mergingStages.size(), 1);
    ASSERT(dynamic_cast<DocumentSourceLimit*>(
        sort()->distributedPlanLogic()->mergingStages.begin()->get()));
}

TEST_F(DocumentSourceSortTest, ParseableSerialization) {
    // Test that SerializationOptions.serializeForCloning works as expected for $sort stage.
    auto expCtx = getExpCtx();
    createSort(BSON("a" << 1));

    ASSERT(!sort()->getLimit());
    DocumentSourceContainer container;
    container.push_back(sort());

    container.push_back(DocumentSourceLimit::create(expCtx, 2));
    sort()->optimizeAt(container.begin(), &container);
    ASSERT_EQUALS(container.size(), 1U);
    ASSERT_EQUALS(*sort()->getLimit(), 2);

    vector<Value> arr;
    sort()->serializeToArray(arr, SerializationOptions{.serializeForCloning = true});
    ASSERT_EQUALS(arr.size(), 1U);
    ASSERT_VALUE_EQ(arr[0], Value(fromjson("{$sort: {a: 1, $_internalLimit: 2}}")));

    // Test that the serialized spec can be parsed to construct a clone of the DocumentSourceSort
    // instance.
    BSONObj spec = arr[0].getDocument().toBson();
    BSONElement specElement = spec.firstElement();
    auto sort2Source = DocumentSourceSort::createFromBson(specElement, expCtx);
    auto sort2 = dynamic_cast<DocumentSourceSort*>(sort2Source.get());
    ASSERT_EQUALS(*sort2->getLimit(), 2);

    arr.clear();
    sort2->serializeToArray(arr, SerializationOptions{.serializeForCloning = true});
    ASSERT_EQUALS(arr.size(), 1U);
    ASSERT_VALUE_EQ(arr[0], Value(fromjson("{$sort: {a: 1, $_internalLimit: 2}}")));
}

TEST_F(DocumentSourceSortTest, QueryShapeSerializationOmitsInternalField) {
    // Test that the serialized spec omits _internalOutputSortKey when serializing for query shape.
    auto expCtx = getExpCtx();
    auto sort = DocumentSourceSort::create(
        expCtx, {BSON("a" << 1), expCtx}, {.outputSortKeyMetadata = true});
    vector<Value> arr;
    sort->serializeToArray(arr, SerializationOptions::kRepresentativeQueryShapeSerializeOptions);
    ASSERT_VALUE_EQ(arr[0], Value{fromjson("{$sort: {a: 1}}")});
}

TEST_F(DocumentSourceSortTest, DoesNotPushProjectBeforeSelf) {
    DocumentSourceContainer container;
    createSort(BSON("_id" << 1));
    auto project =
        DocumentSourceProject::create(BSON("fullDocument" << true), getExpCtx(), "$project"_sd);

    container.push_back(sort());
    container.push_back(project);

    sort()->optimizeAt(container.begin(), &container);

    ASSERT_EQUALS(2U, container.size());
    ASSERT(dynamic_cast<DocumentSourceSort*>(container.begin()->get()));
    ASSERT(dynamic_cast<DocumentSourceSingleDocumentTransformation*>(
        std::next(container.begin())->get()));
}

TEST_F(DocumentSourceSortTest, Dependencies) {
    createSort(BSON("a" << 1 << "b.c" << -1));
    DepsTracker dependencies;
    ASSERT_EQUALS(DepsTracker::State::SEE_NEXT, sort()->getDependencies(&dependencies));
    ASSERT_EQUALS(2U, dependencies.fields.size());
    ASSERT_EQUALS(1U, dependencies.fields.count("a"));
    ASSERT_EQUALS(1U, dependencies.fields.count("b.c"));
    ASSERT_EQUALS(false, dependencies.needWholeDocument);
    ASSERT_EQUALS(false, dependencies.getNeedsMetadata(DocumentMetadataFields::kTextScore));
}

TEST_F(DocumentSourceSortTest, ReportsNoPathsModified) {
    createSort(BSON("a" << 1 << "b.c" << -1));
    auto modifiedPaths = sort()->getModifiedPaths();
    ASSERT(modifiedPaths.type == DocumentSource::GetModPathsReturn::Type::kFiniteSet);
    ASSERT_EQUALS(0U, modifiedPaths.paths.size());
}

TEST_F(DocumentSourceSortTest, AllowsSortOnMetaGeoNearDistance) {
    BSONObj spec = BSON("$sort" << BSON("dist" << BSON("$meta" << "geoNearDistance")));
    BSONElement specElement = spec.firstElement();
    auto sort = DocumentSourceSort::createFromBson(specElement, getExpCtx());

    vector<Value> arr;
    sort->serializeToArray(arr);
    ASSERT_BSONOBJ_EQ(arr[0].getDocument().toBson(),
                      BSON("$sort" << BSON("$computed0" << BSON("$meta" << "geoNearDistance"))));
}

TEST_F(DocumentSourceSortTest, DetectsDependencyOnMeta) {
    BSONObj spec = BSON("$sort" << BSON("dist" << BSON("$meta" << "geoNearDistance")));
    BSONElement specElement = spec.firstElement();
    auto sort = DocumentSourceSort::createFromBson(specElement, getExpCtx());

    DepsTracker dependencies;
    ASSERT_EQUALS(DepsTracker::State::SEE_NEXT, sort->getDependencies(&dependencies));
    ASSERT_EQUALS(0U, dependencies.fields.size());
    ASSERT_EQUALS(true, dependencies.getNeedsMetadata(DocumentMetadataFields::kGeoNearDist));
}

class DocumentSourceSortExecutionTest : public DocumentSourceSortTest {
public:
    void checkResults(deque<DocumentSource::GetNextResult> inputDocs,
                      DocumentSourceSort* sort,
                      string expectedResultSetString) {
        auto source = DocumentSourceMock::createForTest(inputDocs, getExpCtx());
        sort->setSource(source.get());

        // Load the results from the DocumentSourceUnwind.
        vector<Document> resultSet;
        for (auto output = sort->getNext(); output.isAdvanced(); output = sort->getNext()) {
            // Get the current result.
            resultSet.push_back(output.releaseDocument());
        }
        // Verify the DocumentSourceUnwind is exhausted.
        assertEOF();

        // Convert results to BSON once they all have been retrieved (to detect any errors
        // resulting from incorrectly shared sub objects).
        BSONArrayBuilder bsonResultSet;
        for (auto&& result : resultSet) {
            bsonResultSet << result;
        }
        // Check the result set.
        ASSERT_BSONOBJ_EQ(expectedResultSet(expectedResultSetString), bsonResultSet.arr());
    }

protected:
    virtual BSONObj expectedResultSet(string expectedResultSetString) {
        BSONObj wrappedResult =
            // fromjson cannot parse an array, so place the array within an object.
            fromjson(string("{'':") + expectedResultSetString + "}");
        return wrappedResult[""].embeddedObject().getOwned();
    }
};

TEST_F(DocumentSourceSortExecutionTest, ShouldGiveNoOutputIfGivenNoInputs) {
    createSort(BSON("a" << 1));
    checkResults({}, sort(), "[]");
}

TEST_F(DocumentSourceSortExecutionTest, ShouldGiveOneOutputIfGivenOneInput) {
    createSort(BSON("a" << 1));
    checkResults({Document{{"_id", 0}, {"a", 1}}}, sort(), "[{_id:0,a:1}]");
}

TEST_F(DocumentSourceSortExecutionTest, ShouldSortTwoInputsAccordingToOneFieldAscending) {
    createSort(BSON("a" << 1));
    checkResults({Document{{"_id", 0}, {"a", 2}}, Document{{"_id", 1}, {"a", 1}}},
                 sort(),
                 "[{_id:1,a:1},{_id:0,a:2}]");
}

/** Sort spec with a descending field. */
TEST_F(DocumentSourceSortExecutionTest, DescendingOrder) {
    createSort(BSON("a" << -1));
    checkResults({Document{{"_id", 0}, {"a", 2}}, Document{{"_id", 1}, {"a", 1}}},
                 sort(),
                 "[{_id:0,a:2},{_id:1,a:1}]");
}

/** Sort spec with a dotted field. */
TEST_F(DocumentSourceSortExecutionTest, DottedSortField) {
    createSort(BSON("a.b" << 1));
    checkResults({Document{{"_id", 0}, {"a", Document{{"b", 2}}}},
                  Document{{"_id", 1}, {"a", Document{{"b", 1}}}}},
                 sort(),
                 "[{_id:1,a:{b:1}},{_id:0,a:{b:2}}]");
}

/** Sort spec with a compound key. */
TEST_F(DocumentSourceSortExecutionTest, CompoundSortSpec) {
    createSort(BSON("a" << 1 << "b" << 1));
    checkResults({Document{{"_id", 0}, {"a", 1}, {"b", 3}},
                  Document{{"_id", 1}, {"a", 1}, {"b", 2}},
                  Document{{"_id", 2}, {"a", 0}, {"b", 4}}},
                 sort(),
                 "[{_id:2,a:0,b:4},{_id:1,a:1,b:2},{_id:0,a:1,b:3}]");
}

/** Sort spec with a compound key and descending order. */
TEST_F(DocumentSourceSortExecutionTest, CompoundSortSpecAlternateOrder) {
    createSort(BSON("a" << -1 << "b" << 1));
    checkResults({Document{{"_id", 0}, {"a", 1}, {"b", 3}},
                  Document{{"_id", 1}, {"a", 1}, {"b", 2}},
                  Document{{"_id", 2}, {"a", 0}, {"b", 4}}},
                 sort(),
                 "[{_id:1,a:1,b:2},{_id:0,a:1,b:3},{_id:2,a:0,b:4}]");
}

/** Sort spec with a compound key and descending order. */
TEST_F(DocumentSourceSortExecutionTest, CompoundSortSpecAlternateOrderSecondField) {
    createSort(BSON("a" << 1 << "b" << -1));
    checkResults({Document{{"_id", 0}, {"a", 1}, {"b", 3}},
                  Document{{"_id", 1}, {"a", 1}, {"b", 2}},
                  Document{{"_id", 2}, {"a", 0}, {"b", 4}}},
                 sort(),
                 "[{_id:2,a:0,b:4},{_id:0,a:1,b:3},{_id:1,a:1,b:2}]");
}

/** Sorting different types is not supported. */
TEST_F(DocumentSourceSortExecutionTest, InconsistentTypeSort) {
    createSort(BSON("a" << 1));
    checkResults({Document{{"_id", 0}, {"a", 1}}, Document{{"_id", 1}, {"a", "foo"_sd}}},
                 sort(),
                 "[{_id:0,a:1},{_id:1,a:\"foo\"}]");
}

/** Sorting different numeric types is supported. */
TEST_F(DocumentSourceSortExecutionTest, MixedNumericSort) {
    createSort(BSON("a" << 1));
    checkResults({Document{{"_id", 0}, {"a", 2.3}}, Document{{"_id", 1}, {"a", 1}}},
                 sort(),
                 "[{_id:1,a:1},{_id:0,a:2.3}]");
}

/** Ordering of a missing value. */
TEST_F(DocumentSourceSortExecutionTest, MissingValue) {
    createSort(BSON("a" << 1));
    checkResults(
        {Document{{"_id", 0}, {"a", 1}}, Document{{"_id", 1}}}, sort(), "[{_id:1},{_id:0,a:1}]");
}

/** Ordering of a null value. */
TEST_F(DocumentSourceSortExecutionTest, NullValue) {
    createSort(BSON("a" << 1));
    checkResults({Document{{"_id", 0}, {"a", 1}}, Document{{"_id", 1}, {"a", BSONNULL}}},
                 sort(),
                 "[{_id:1,a:null},{_id:0,a:1}]");
}

/**
 * Order by text score.
 */
TEST_F(DocumentSourceSortExecutionTest, TextScore) {
    MutableDocument first(Document{{"_id", 0}});
    first.metadata().setTextScore(10);
    MutableDocument second(Document{{"_id", 1}});
    second.metadata().setTextScore(20);

    createSort(BSON("$computed0" << metaTextScore));
    checkResults({first.freeze(), second.freeze()}, sort(), "[{_id:1},{_id:0}]");
}

/**
 * Order by random value in metadata.
 */
TEST_F(DocumentSourceSortExecutionTest, RandMeta) {
    MutableDocument first(Document{{"_id", 0}});
    first.metadata().setRandVal(0.01);
    MutableDocument second(Document{{"_id", 1}});
    second.metadata().setRandVal(0.02);

    createSort(BSON("$computed0" << BSON("$meta" << "randVal")));
    checkResults({first.freeze(), second.freeze()}, sort(), "[{_id:1},{_id:0}]");
}

/**
 * Sort tasserts on control event.
 */
DEATH_TEST_REGEX_F(DocumentSourceSortExecutionTest,
                   CannotHandleControlEvent,
                   "Tripwire assertion.*10358905") {
    MutableDocument doc(Document{{"_id", 0}});
    doc.metadata().setChangeStreamControlEvent();

    createSort(BSON("_id" << 1));
    auto source = DocumentSourceMock::createForTest({doc.freeze()}, getExpCtx());
    sort()->setSource(source.get());
    ASSERT_THROWS_CODE(sort()->getNext(), AssertionException, 10358905);
}

/** A missing nested object within an array returns an empty array. */
TEST_F(DocumentSourceSortExecutionTest, MissingObjectWithinArray) {
    createSort(BSON("a.b" << 1));
    checkResults({Document{{"_id", 0}, {"a", DOC_ARRAY(1)}},
                  Document{{"_id", 1}, {"a", DOC_ARRAY(DOC("b" << 1))}}},
                 sort(),
                 "[{_id:0,a:[1]},{_id:1,a:[{b:1}]}]");
}

/** Compare nested values from within an array. */
TEST_F(DocumentSourceSortExecutionTest, ExtractArrayValues) {
    createSort(BSON("a.b" << 1));
    checkResults({Document{{"_id", 0}, {"a", DOC_ARRAY(DOC("b" << 1) << DOC("b" << 2))}},
                  Document{{"_id", 1}, {"a", DOC_ARRAY(DOC("b" << 1) << DOC("b" << 0))}}},
                 sort(),
                 "[{_id:1,a:[{b:1},{b:0}]},{_id:0,a:[{b:1},{b:2}]}]");
}

TEST_F(DocumentSourceSortExecutionTest, ParseableSerialization) {
    // Test that the serialized spec created with SerializationOptions.serializeForCloning option
    // can be parsed to construct a clone of the DocumentSourceSort instance.
    auto expCtx = getExpCtx();

    BSONObj spec = fromjson("{$sort: {a: 1, $_internalLimit: 2}}");
    BSONElement specElement = spec.firstElement();
    setSort(DocumentSourceSort::createFromBson(specElement, getExpCtx()));
    ASSERT_EQUALS(*sort()->getLimit(), 2);
    checkResults({Document{{"_id", 0}, {"a", 1}},
                  Document{{"_id", 1}, {"a", 2}},
                  Document{{"_id", 2}, {"a", 3}}},
                 sort(),
                 "[{_id:0,a:1},{_id:1,a:2}]");

    vector<Value> arr;
    sort()->serializeToArray(arr, SerializationOptions{.serializeForCloning = true});
    ASSERT_EQUALS(arr.size(), 1U);
    ASSERT_VALUE_EQ(arr[0], Value(fromjson("{$sort: {a: 1, $_internalLimit: 2}}")));
}

TEST_F(DocumentSourceSortExecutionTest, FailsGracefullyWithInvalidInternalParameters) {
    auto expCtx = getExpCtx();

    auto spec = fromjson("{$sort: {a: 1, $_internalLimit: 'surprise string!'}}");
    ASSERT_THROWS_CODE(DocumentSourceSort::createFromBson(spec.firstElement(), expCtx),
                       AssertionException,
                       9028702);

    spec = fromjson("{$sort: {a: 1, $_internalOutputSortKey: 'surprise string!'}}");
    ASSERT_THROWS_CODE(
        DocumentSourceSort::createFromBson(spec.firstElement(), expCtx), AssertionException, 15974);

    // Must be a boolean, not bool-"ish".
    spec = fromjson("{$sort: {a: 1, $_internalOutputSortKey: 1}}");
    ASSERT_THROWS_CODE(
        DocumentSourceSort::createFromBson(spec.firstElement(), expCtx), AssertionException, 16410);
}


TEST_F(DocumentSourceSortExecutionTest, ShouldPauseWhenAskedTo) {
    auto sort = DocumentSourceSort::create(getExpCtx(), BSON("a" << 1));
    auto mock =
        DocumentSourceMock::createForTest({DocumentSource::GetNextResult::makePauseExecution(),
                                           Document{{"a", 0}},
                                           DocumentSource::GetNextResult::makePauseExecution()},
                                          getExpCtx());
    sort->setSource(mock.get());

    // Should propagate the first pause.
    ASSERT_TRUE(sort->getNext().isPaused());

    // Should load the single document, then pause.
    ASSERT_TRUE(sort->getNext().isPaused());

    // Now it should start giving results.
    auto result = sort->getNext();
    ASSERT_TRUE(result.isAdvanced());
    ASSERT_DOCUMENT_EQ(result.releaseDocument(), (Document{{"a", 0}}));
}

TEST_F(DocumentSourceSortExecutionTest, ShouldResumePopulationBetweenPauses) {
    auto sort = DocumentSourceSort::create(getExpCtx(), BSON("a" << 1));
    auto mock =
        DocumentSourceMock::createForTest({Document{{"a", 1}},
                                           DocumentSource::GetNextResult::makePauseExecution(),
                                           Document{{"a", 0}}},
                                          getExpCtx());
    sort->setSource(mock.get());

    // Should load the first document, then propagate the pause.
    ASSERT_TRUE(sort->getNext().isPaused());

    // Should finish loading and start yielding results in sorted order.
    auto result = sort->getNext();
    ASSERT_TRUE(result.isAdvanced());
    ASSERT_DOCUMENT_EQ(result.releaseDocument(), (Document{{"a", 0}}));

    result = sort->getNext();
    ASSERT_TRUE(result.isAdvanced());
    ASSERT_DOCUMENT_EQ(result.releaseDocument(), (Document{{"a", 1}}));

    ASSERT_TRUE(sort->getNext().isEOF());
    ASSERT_TRUE(sort->getNext().isEOF());
    ASSERT_TRUE(sort->getNext().isEOF());
}

std::pair<boost::intrusive_ptr<DocumentSourceMock>, boost::intrusive_ptr<DocumentSourceSort>>
initSpillingTest(boost::intrusive_ptr<ExpressionContext> expCtx,
                 const unittest::TempDir& tempDir,
                 size_t maxMemoryUsageBytes,
                 size_t largeStrSize) {
    // Allow the $sort stage to spill to disk.
    expCtx->setTempDir(tempDir.path());
    expCtx->setAllowDiskUse(true);
    auto sort = DocumentSourceSort::create(
        expCtx, {BSON("_id" << -1), expCtx}, {.maxMemoryUsageBytes = maxMemoryUsageBytes});

    string largeStr(largeStrSize, 'x');
    auto mock =
        DocumentSourceMock::createForTest({Document{{"_id", 0}, {"largeStr", largeStr}},
                                           DocumentSource::GetNextResult::makePauseExecution(),
                                           Document{{"_id", 1}, {"largeStr", largeStr}},
                                           DocumentSource::GetNextResult::makePauseExecution(),
                                           Document{{"_id", 2}, {"largeStr", largeStr}}},
                                          expCtx);
    sort->setSource(mock.get());
    return {std::move(mock), std::move(sort)};
}

void assertSpillingTestReturn(boost::intrusive_ptr<DocumentSourceSort> sort) {
    auto next = sort->getNext();
    ASSERT_TRUE(next.isAdvanced());
    ASSERT_VALUE_EQ(next.releaseDocument()["_id"], Value(2));

    next = sort->getNext();
    ASSERT_TRUE(next.isAdvanced());
    ASSERT_VALUE_EQ(next.releaseDocument()["_id"], Value(1));

    next = sort->getNext();
    ASSERT_TRUE(next.isAdvanced());
    ASSERT_VALUE_EQ(next.releaseDocument()["_id"], Value(0));

    ASSERT_TRUE(sort->getNext().isEOF());
}

TEST_F(DocumentSourceSortExecutionTest, ShouldBeAbleToPauseLoadingWhileSpilled) {
    unittest::TempDir tempDir("DocumentSourceSortTest");
    auto [mock, sort] = initSpillingTest(getExpCtx(), tempDir, 1000000, 1000000);

    // There were 2 pauses, so we should expect 2 paused results before any results can be returned.
    ASSERT_TRUE(sort->getNext().isPaused());
    ASSERT_TRUE(sort->getNext().isPaused());

    assertSpillingTestReturn(sort);

    const auto* sortStats = static_cast<const SortStats*>(sort->getSpecificStats());
    ASSERT_EQ(sortStats->spillingStats.getSpills(), 3);
    ASSERT_EQ(sortStats->spillingStats.getSpilledRecords(), 3);
    ASSERT_EQ(sortStats->spillingStats.getSpilledBytes(), 3000099);
    ASSERT_LT(sortStats->spillingStats.getSpilledDataStorageSize(), 3000000);
    ASSERT_GT(sortStats->spillingStats.getSpilledDataStorageSize(), 0);
}

TEST_F(DocumentSourceSortExecutionTest, ShouldBeAbleToManuallySpillBeforeReturingFirstDocument) {
    unittest::TempDir tempDir("DocumentSourceSortTest");
    auto [mock, sort] = initSpillingTest(getExpCtx(), tempDir, 1000000, 10);

    ASSERT_TRUE(sort->getNext().isPaused());
    ASSERT_TRUE(sort->getNext().isPaused());

    sort->forceSpill();
    assertSpillingTestReturn(sort);

    const auto* sortStats = static_cast<const SortStats*>(sort->getSpecificStats());
    ASSERT_EQ(sortStats->spillingStats.getSpills(), 2);
    ASSERT_EQ(sortStats->spillingStats.getSpilledRecords(), 3);
    ASSERT_EQ(sortStats->spillingStats.getSpilledBytes(), 129);
    ASSERT_LT(sortStats->spillingStats.getSpilledDataStorageSize(), 200);
    ASSERT_GT(sortStats->spillingStats.getSpilledDataStorageSize(), 0);
}

TEST_F(DocumentSourceSortExecutionTest, ShouldBeAbleToManuallySpillAfterReturingFirstDocument) {
    unittest::TempDir tempDir("DocumentSourceSortTest");
    auto [mock, sort] = initSpillingTest(getExpCtx(), tempDir, 1000000, 10);

    ASSERT_TRUE(sort->getNext().isPaused());
    ASSERT_TRUE(sort->getNext().isPaused());

    auto next = sort->getNext();
    ASSERT_TRUE(next.isAdvanced());
    ASSERT_VALUE_EQ(next.releaseDocument()["_id"], Value(2));

    sort->forceSpill();

    next = sort->getNext();
    ASSERT_TRUE(next.isAdvanced());
    ASSERT_VALUE_EQ(next.releaseDocument()["_id"], Value(1));

    next = sort->getNext();
    ASSERT_TRUE(next.isAdvanced());
    ASSERT_VALUE_EQ(next.releaseDocument()["_id"], Value(0));

    ASSERT_TRUE(sort->getNext().isEOF());

    const auto* sortStats = static_cast<const SortStats*>(sort->getSpecificStats());
    ASSERT_EQ(sortStats->spillingStats.getSpills(), 1);
    ASSERT_EQ(sortStats->spillingStats.getSpilledRecords(), 2);
    ASSERT_EQ(sortStats->spillingStats.getSpilledBytes(), 86);
    ASSERT_LT(sortStats->spillingStats.getSpilledDataStorageSize(), 200);
    ASSERT_GT(sortStats->spillingStats.getSpilledDataStorageSize(), 0);
}

TEST_F(DocumentSourceSortExecutionTest, ShouldBeAbleToReportSpillingStatsInBoundedSort) {
    RAIIServerParameterControllerForTest sortMemoryLimit{
        "internalQueryMaxBlockingSortMemoryUsageBytes", 3 * 1024};

    unittest::TempDir tempDir("DocumentSourceSortTest");
    auto expCtx = getExpCtx();
    expCtx->setTempDir(tempDir.path());
    expCtx->setAllowDiskUse(true);
    auto sort = DocumentSourceSort::createBoundedSort({BSON("time" << 1), expCtx},
                                                      "min",
                                                      -1,
                                                      boost::none /*limit*/,
                                                      false /*outputSortKeyMetadata*/,
                                                      expCtx);

    std::string largeStr(1024, 'x');

    std::vector<Document> data = {
        Document{{"time", Date_t::fromMillisSinceEpoch(1)}, {"largeStr", largeStr}},
        Document{{"time", Date_t::fromMillisSinceEpoch(0)}, {"largeStr", largeStr}},
        Document{{"time", Date_t::fromMillisSinceEpoch(2)}, {"largeStr", largeStr}},
        Document{{"time", Date_t::fromMillisSinceEpoch(3)}, {"largeStr", largeStr}}};
    for (auto& doc : data) {
        MutableDocument mdoc{doc};
        DocumentMetadataFields metadata;
        metadata.setTimeseriesBucketMinTime(doc.getField("time").getDate());
        mdoc.setMetadata(std::move(metadata));
        doc = mdoc.freeze();
    }

    auto mock = DocumentSourceMock::createForTest(std::move(data), expCtx);
    sort->setSource(mock.get());

    auto next = sort->getNext();
    ASSERT_TRUE(next.isAdvanced());
    ASSERT_VALUE_EQ(next.releaseDocument()["time"], Value(Date_t::fromMillisSinceEpoch(0)));

    next = sort->getNext();
    ASSERT_TRUE(next.isAdvanced());
    ASSERT_VALUE_EQ(next.releaseDocument()["time"], Value(Date_t::fromMillisSinceEpoch(1)));

    next = sort->getNext();
    ASSERT_TRUE(next.isAdvanced());
    ASSERT_VALUE_EQ(next.releaseDocument()["time"], Value(Date_t::fromMillisSinceEpoch(2)));

    next = sort->getNext();
    ASSERT_TRUE(next.isAdvanced());
    ASSERT_VALUE_EQ(next.releaseDocument()["time"], Value(Date_t::fromMillisSinceEpoch(3)));

    ASSERT_TRUE(sort->getNext().isEOF());

    const auto* sortStats = static_cast<const SortStats*>(sort->getSpecificStats());
    ASSERT_EQ(sortStats->spillingStats.getSpills(), 2);
    ASSERT_EQ(sortStats->spillingStats.getSpilledRecords(), 4);
    ASSERT_EQ(sortStats->spillingStats.getSpilledBytes(), 4296);
    ASSERT_LT(sortStats->spillingStats.getSpilledDataStorageSize(), 1024);
    ASSERT_GT(sortStats->spillingStats.getSpilledDataStorageSize(), 0);
}

TEST_F(DocumentSourceSortExecutionTest,
       ShouldBeAbleToReportSpillingStatsInBoundedSortWithSortKeyMetadata) {
    RAIIServerParameterControllerForTest sortMemoryLimit{
        "internalQueryMaxBlockingSortMemoryUsageBytes", 3 * 1024};

    unittest::TempDir tempDir("DocumentSourceSortTest");
    auto expCtx = getExpCtx();
    expCtx->setTempDir(tempDir.path());
    expCtx->setAllowDiskUse(true);
    auto sort = DocumentSourceSort::createBoundedSort({BSON("time" << 1), expCtx},
                                                      "min",
                                                      -1,
                                                      boost::none /*limit*/,
                                                      true /*outputSortKeyMetadata*/,
                                                      expCtx);

    std::string largeStr(1024, 'x');

    std::vector<Document> data = {
        Document{{"time", Date_t::fromMillisSinceEpoch(1)}, {"largeStr", largeStr}},
        Document{{"time", Date_t::fromMillisSinceEpoch(0)}, {"largeStr", largeStr}},
        Document{{"time", Date_t::fromMillisSinceEpoch(2)}, {"largeStr", largeStr}},
        Document{{"time", Date_t::fromMillisSinceEpoch(3)}, {"largeStr", largeStr}}};
    for (auto& doc : data) {
        MutableDocument mdoc{doc};
        DocumentMetadataFields metadata;
        metadata.setTimeseriesBucketMinTime(doc.getField("time").getDate());
        mdoc.setMetadata(std::move(metadata));
        doc = mdoc.freeze();
    }

    auto mock = DocumentSourceMock::createForTest(std::move(data), expCtx);
    sort->setSource(mock.get());

    auto next = sort->getNext();
    ASSERT_TRUE(next.isAdvanced());
    ASSERT_VALUE_EQ(next.releaseDocument()["time"], Value(Date_t::fromMillisSinceEpoch(0)));

    next = sort->getNext();
    ASSERT_TRUE(next.isAdvanced());
    ASSERT_VALUE_EQ(next.releaseDocument()["time"], Value(Date_t::fromMillisSinceEpoch(1)));

    next = sort->getNext();
    ASSERT_TRUE(next.isAdvanced());
    ASSERT_VALUE_EQ(next.releaseDocument()["time"], Value(Date_t::fromMillisSinceEpoch(2)));

    next = sort->getNext();
    ASSERT_TRUE(next.isAdvanced());
    ASSERT_VALUE_EQ(next.releaseDocument()["time"], Value(Date_t::fromMillisSinceEpoch(3)));

    ASSERT_TRUE(sort->getNext().isEOF());

    const auto* sortStats = static_cast<const SortStats*>(sort->getSpecificStats());
    ASSERT_EQ(sortStats->spillingStats.getSpills(), 2);
    ASSERT_EQ(sortStats->spillingStats.getSpilledRecords(), 4);
    ASSERT_EQ(sortStats->spillingStats.getSpilledBytes(), 4340);
    ASSERT_LT(sortStats->spillingStats.getSpilledDataStorageSize(), 1024);
    ASSERT_GT(sortStats->spillingStats.getSpilledDataStorageSize(), 0);
}

TEST_F(DocumentSourceSortExecutionTest,
       ShouldErrorIfNotAllowedToSpillToDiskAndResultSetIsTooLarge) {
    auto expCtx = getExpCtx();
    expCtx->setAllowDiskUse(false);
    const size_t maxMemoryUsageBytes = 1000;

    auto sort = DocumentSourceSort::create(
        expCtx, {BSON("_id" << -1), expCtx}, {.maxMemoryUsageBytes = maxMemoryUsageBytes});

    string largeStr(maxMemoryUsageBytes, 'x');
    auto mock = DocumentSourceMock::createForTest({Document{{"_id", 0}, {"largeStr", largeStr}},
                                                   Document{{"_id", 1}, {"largeStr", largeStr}}},
                                                  expCtx);
    sort->setSource(mock.get());

    ASSERT_THROWS_CODE(
        sort->getNext(), AssertionException, ErrorCodes::QueryExceededMemoryLimitNoDiskUseAllowed);
}

TEST_F(DocumentSourceSortExecutionTest, ShouldCorrectlyTrackMemoryUsageBetweenPauses) {
    auto expCtx = getExpCtx();
    expCtx->setAllowDiskUse(false);
    const size_t maxMemoryUsageBytes = 1000;

    auto sort = DocumentSourceSort::create(
        expCtx, {BSON("_id" << -1), expCtx}, {.maxMemoryUsageBytes = maxMemoryUsageBytes});

    string largeStr(maxMemoryUsageBytes / 5, 'x');
    auto mock =
        DocumentSourceMock::createForTest({Document{{"_id", 0}, {"largeStr", largeStr}},
                                           DocumentSource::GetNextResult::makePauseExecution(),
                                           Document{{"_id", 1}, {"largeStr", largeStr}},
                                           Document{{"_id", 2}, {"largeStr", largeStr}}},
                                          expCtx);
    sort->setSource(mock.get());

    // The first getNext() should pause.
    ASSERT_TRUE(sort->getNext().isPaused());

    // The next should realize it's used too much memory.
    ASSERT_THROWS_CODE(
        sort->getNext(), AssertionException, ErrorCodes::QueryExceededMemoryLimitNoDiskUseAllowed);
}

TEST_F(DocumentSourceSortTest, Redaction) {
    createSort(BSON("a" << 1));
    auto boundedSort = DocumentSourceSort::createBoundedSort(
        sort()->getSortKeyPattern(), DocumentSourceSort::kMin, 1337, 10, false, getExpCtx());

    ASSERT_BSONOBJ_EQ_AUTO(  //
        R"({"$sort":{"HASH<a>":1}})",
        redact(*sort(), true));
    ASSERT_BSONOBJ_EQ_AUTO(  // NOLINT
        R"({
            "$_internalBoundedSort": {
                "sortKey": {
                    "HASH<a>": 1
                },
                "bound": {
                    "base": "min",
                    "offsetSeconds": "?number"
                },
                "limit": "?number"
            }
        })",
        redact(*boundedSort, true));

    ASSERT_BSONOBJ_EQ_AUTO(  //
        R"({"$sort":{"sortKey":{"HASH<a>":1}}})",
        redact(*sort(), true, ExplainOptions::Verbosity::kQueryPlanner));

    ASSERT_BSONOBJ_EQ_AUTO(  // NOLINT
        R"({
            "$_internalBoundedSort": {
                "sortKey": {
                    "HASH<a>": 1
                },
                "bound": {
                    "base": "min",
                    "offsetSeconds": "?number"
                },
                "limit": "?number"
            }
        })",
        redact(*boundedSort, true, ExplainOptions::Verbosity::kQueryPlanner));

    ASSERT_BSONOBJ_EQ_AUTO(  //
        R"({
            "$sort": {
                "sortKey": {
                    "HASH<a>": 1
                }
            },
            "totalDataSizeSortedBytesEstimate": "?number",
            "usedDisk": "?bool",
            "spills": "?number",
            "spilledDataStorageSize": "?number"
        })",
        redact(*sort(), true, ExplainOptions::Verbosity::kExecStats));

    ASSERT_BSONOBJ_EQ_AUTO(  // NOLINT
        R"({
            "$_internalBoundedSort": {
                "sortKey": {
                    "HASH<a>": 1
                },
                "bound": {
                    "base": "min",
                    "offsetSeconds": "?number"
                },
                "limit": "?number"
            },
            "totalDataSizeSortedBytesEstimate": "?number",
            "usedDisk": "?bool",
            "spills": "?number",
            "spilledDataStorageSize": "?number"
        })",
        redact(*boundedSort, true, ExplainOptions::Verbosity::kExecStats));
}

TEST_F(DocumentSourceSortTest, RedactionWithSortKeyMetadata) {
    createSort(BSON("a" << 1));
    auto boundedSort = DocumentSourceSort::createBoundedSort(
        sort()->getSortKeyPattern(), DocumentSourceSort::kMin, 1337, 10, true, getExpCtx());

    ASSERT_BSONOBJ_EQ_AUTO(  //
        R"({"$sort":{"HASH<a>":1}})",
        redact(*sort(), true));
    ASSERT_BSONOBJ_EQ_AUTO(  // NOLINT
        R"({
            "$_internalBoundedSort": {
                "sortKey": {
                    "HASH<a>": 1
                },
                "bound": {
                    "base": "min",
                    "offsetSeconds": "?number"
                },
                "limit": "?number"
            }
        })",
        redact(*boundedSort, true));

    ASSERT_BSONOBJ_EQ_AUTO(  //
        R"({"$sort":{"sortKey":{"HASH<a>":1}}})",
        redact(*sort(), true, ExplainOptions::Verbosity::kQueryPlanner));

    ASSERT_BSONOBJ_EQ_AUTO(  // NOLINT
        R"({
            "$_internalBoundedSort": {
                "sortKey": {
                    "HASH<a>": 1
                },
                "bound": {
                    "base": "min",
                    "offsetSeconds": "?number"
                },
                "limit": "?number"
            }
        })",
        redact(*boundedSort, true, ExplainOptions::Verbosity::kQueryPlanner));

    ASSERT_BSONOBJ_EQ_AUTO(  //
        R"({
            "$sort": {
                "sortKey": {
                    "HASH<a>": 1
                }
            },
            "totalDataSizeSortedBytesEstimate": "?number",
            "usedDisk": "?bool",
            "spills": "?number",
            "spilledDataStorageSize": "?number"
        })",
        redact(*sort(), true, ExplainOptions::Verbosity::kExecStats));

    ASSERT_BSONOBJ_EQ_AUTO(  // NOLINT
        R"({
            "$_internalBoundedSort": {
                "sortKey": {
                    "HASH<a>": 1
                },
                "bound": {
                    "base": "min",
                    "offsetSeconds": "?number"
                },
                "limit": "?number"
            },
            "totalDataSizeSortedBytesEstimate": "?number",
            "usedDisk": "?bool",
            "spills": "?number",
            "spilledDataStorageSize": "?number"
        })",
        redact(*boundedSort, true, ExplainOptions::Verbosity::kExecStats));
}

void assertProducesSortKeyMetadata(auto expCtx, auto sortStage) {
    const auto mock =
        DocumentSourceMock::createForTest({Document{{"_id", 0}}, Document{{"_id", 1}}}, expCtx);
    auto stage = exec::agg::buildStage(sortStage);
    stage->setSource(mock.get());
    const auto output1 = stage->getNext();
    ASSERT(output1.isAdvanced());
    ASSERT(output1.getDocument().metadata().hasSortKey());
    const auto output2 = stage->getNext();
    ASSERT(output2.isAdvanced());
    ASSERT(output2.getDocument().metadata().hasSortKey());
    ASSERT(stage->getNext().isEOF());
}

TEST_F(DocumentSourceSortExecutionTest, ShouldOutputSortKeyMetadataIfRequested) {
    auto expCtx = getExpCtx();
    auto sort = DocumentSourceSort::create(
        expCtx, {BSON("_id" << -1), expCtx}, {.outputSortKeyMetadata = true});
    assertProducesSortKeyMetadata(expCtx, sort);
}

TEST_F(DocumentSourceSortExecutionTest, OutputSortKeyHoldsUpOverCloning) {
    auto expCtx = getExpCtx();
    auto sortBase = DocumentSourceSort::create(
        expCtx, {BSON("_id" << -1), expCtx}, {.outputSortKeyMetadata = true});
    auto clone = sortBase->clone(expCtx);
    assertProducesSortKeyMetadata(expCtx, clone);
}

TEST_F(DocumentSourceSortExecutionTest, OutputSortKeyHoldsUpOverSerialization) {
    auto expCtx = getExpCtx();
    auto sortBase = DocumentSourceSort::create(
        expCtx, {BSON("_id" << -1), expCtx}, {.outputSortKeyMetadata = true});
    BSONObj serializedBson = [&]() {
        std::vector<Value> serialization;
        sortBase->serializeToArray(serialization);
        ASSERT(serialization.size() == 1);
        return serialization.front().getDocument().toBson();
    }();
    auto reparsed = DocumentSourceSort::createFromBson(serializedBson.firstElement(), expCtx);
    assertProducesSortKeyMetadata(expCtx, reparsed);
}

}  // namespace
}  // namespace mongo
