/**
 *    Copyright (C) 2025-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */
#include "mongo/db/service_context_test_fixture.h"
#include "mongo/db/version_context.h"
#include "mongo/unittest/death_test.h"
#include "mongo/unittest/unittest.h"

namespace mongo {
namespace {
// (Generic FCV reference): used for testing, should exist across LTS binary versions
static const VersionContext kLatestVersionContext{multiversion::GenericFCV::kLatest};
static const VersionContext kLastLTSVersionContext{multiversion::GenericFCV::kLastLTS};

class VersionContextDecorationTest : public ServiceContextTest {
public:
    ServiceContext::UniqueOperationContext opCtxHolder{makeOperationContext()};
    OperationContext* opCtx{opCtxHolder.get()};
};

TEST_F(VersionContextDecorationTest, GetDecorationDefault) {
    ASSERT_EQ(VersionContext{}, VersionContext::getDecoration(opCtx));
}

TEST_F(VersionContextDecorationTest, SetDecoration) {
    {
        ClientLock lk(opCtx->getClient());
        VersionContext::setDecoration(lk, opCtx, kLatestVersionContext);
    }
    ASSERT_EQ(kLatestVersionContext, VersionContext::getDecoration(opCtx));
}

TEST_F(VersionContextDecorationTest, ScopedSetDecoration) {
    ASSERT_EQ(VersionContext{}, VersionContext::getDecoration(opCtx));
    {
        VersionContext::ScopedSetDecoration scopedSetDecoration(opCtx, kLastLTSVersionContext);
        ASSERT_EQ(kLastLTSVersionContext, VersionContext::getDecoration(opCtx));
    }
    ASSERT_EQ(VersionContext{}, VersionContext::getDecoration(opCtx));
}

DEATH_TEST_F(VersionContextDecorationTest,
             ScopedSetDecorationRecursive,
             "Refusing to set a VersionContext on an operation that already has one") {
    VersionContext::ScopedSetDecoration scopedVCtx(opCtx, kLastLTSVersionContext);
    VersionContext::ScopedSetDecoration(opCtx, kLastLTSVersionContext);
}

}  // namespace
}  // namespace mongo
