#!/bin/bash

# This script fetches sources for ICU and builds custom ICU data files (one big-endian data file
# and one-little endian data file).  Both the sources and the data files are trimmed down for size.
#
# This script can be run from anywhere within the root of the source repository.  This script
# expects the ICU third-party directory (src/third_party/icu4c-xx.yy/) to exist and contain a
# newline-separated language file in source/mongo_sources/languages.txt.  This language file must
# list each locale for which collation data should be packaged as part of the generated custom data
# file.
#
# The script accepts a single optional argument, which is the path to the .dat archive to use as the
# source of the trimmed-down ICU .dat files generated as output. If omitted, the .dat archive which
# is included in the ICU source code is used by default. The included .dat file included with some
# versions of ICU does not contain all the collations needed by mongodb. If this script fails while
# checking the data list for collations, it must be invoked with the path to our custom .dat
# archive:
# ./src/third_party/scripts/icu_get_sources.sh \
#      $(pwd)/src/third_party/icu4c-57.1/source/mongo_sources/icudt57l.dat
#
# That archive can be generated by the ICU tool here: http://apps.icu-project.org/datacustom/, you
# should check all the boxes and download the zipfile for the major version of ICU. This script
# will strip out anything that isn't needed.
#
# This script returns a zero exit code on success.

set -euo pipefail
IFS=$'\n\t'

# Expect 0 or 1 arguments.
if [ "$#" -gt 1 ]; then
    echo "$0: too many arguments" >&2
    exit 1
fi

# Set value of original .dat archive from the first argument.
if [ "$#" -eq 1 ]; then
    ORIGINAL_DATA_FILE=${1}
fi

KERNEL="$(uname)"
if [ "$KERNEL" != Linux ]; then
    echo "$0: kernel '$KERNEL' not supported" >&2
    exit 1
fi

NAME=icu4c
MAJOR_VERSION=57
MINOR_VERSION=1
VERSION="57.1"

LIB_GIT_REVISION=d2fbfead0830e1119dad0eee49350fc419cf7eb0
LIB_GIT_REPO=https://github.com/mongodb-forks/icu.git

ICU_THIRD_PARTY_DIR="$(git rev-parse --show-toplevel)/src/third_party/${NAME}-${VERSION}"
MONGO_SOURCES_DIR="${ICU_THIRD_PARTY_DIR}/source/mongo_sources"
LANGUAGE_FILE_IN="${MONGO_SOURCES_DIR}/languages.txt"
ICU_DATA_FILE_LITTLE_ENDIAN_OUT="${MONGO_SOURCES_DIR}/icudt${MAJOR_VERSION}l.dat"
ICU_DATA_FILE_BIG_ENDIAN_OUT="${MONGO_SOURCES_DIR}/icudt${MAJOR_VERSION}b.dat"

#
# Set up temp directory.
#

TEMP_DIR="$(mktemp -d /tmp/icu.XXXXXX)"
trap "rm -rf $TEMP_DIR" EXIT

SOURCE_DIR="${TEMP_DIR}/source"
INSTALL_DIR="${TEMP_DIR}/install"
DATA_DIR="${TEMP_DIR}/data"
FRESH_SOURCE_DIR="${TEMP_DIR}/source2"
mkdir "$SOURCE_DIR" "$INSTALL_DIR" "$DATA_DIR" "$FRESH_SOURCE_DIR"

#
# Clone and checkout the git repo into temp directory.
#

git clone $LIB_GIT_REPO $SOURCE_DIR
git -C $SOURCE_DIR checkout $LIB_GIT_REVISION

#
# Build, test and install ICU in temp directory, in order to use data packaging tools.
#

cd "${SOURCE_DIR}/icu4c/source"
./runConfigureICU "$KERNEL" --prefix="${TEMP_DIR}/install"
make -j
make pcheck
make install

#
# Generate trimmed-down list of data to include in custom data files.
#

# If the original data file wasn't passed as an argument, use the one from the ICU source tree.
if [ -z ${ORIGINAL_DATA_FILE+x} ]; then
    ORIGINAL_DATA_FILE="${SOURCE_DIR}/icu4c/source/data/in/icudt${MAJOR_VERSION}l.dat"
fi

ORIGINAL_DATA_LIST="${DATA_DIR}/icudt${MAJOR_VERSION}l.lst.orig"
NEW_DATA_LIST="${DATA_DIR}/icudt${MAJOR_VERSION}l.lst"

LD_LIBRARY_PATH= eval $("${INSTALL_DIR}/bin/icu-config" --invoke=icupkg) -l "$ORIGINAL_DATA_FILE" \
    > "$ORIGINAL_DATA_LIST"

# Collation data sets
DESIRED_DATA_DIRECTORIES="coll"
BASE_FILES="root.res
ucadata.icu"
for DESIRED_DATA_DIRECTORY in $DESIRED_DATA_DIRECTORIES; do
    for BASE_FILE in $BASE_FILES; do
        echo "Checking $ORIGINAL_DATA_LIST for $DESIRED_DATA_DIRECTORY/$BASE_FILE"
        # Using grep to sanity-check that the file indeed appears in the original data list.
        grep -E "^${DESIRED_DATA_DIRECTORY}/${BASE_FILE}$" "$ORIGINAL_DATA_LIST" >> "$NEW_DATA_LIST"
    done
    for LANGUAGE in $(grep -Ev "^#" "$LANGUAGE_FILE_IN"); do
        # Ditto above.
        echo "Checking $ORIGINAL_DATA_LIST for $DESIRED_DATA_DIRECTORY/$LANGUAGE"
        grep -E "^${DESIRED_DATA_DIRECTORY}/${LANGUAGE}.res$" "$ORIGINAL_DATA_LIST" \
            >> "$NEW_DATA_LIST"
    done
done

# UStringPrepProfile: USPREP_RFC4013_SASLPREP and NFKC normalization.
grep -E "^rfc4013.spp$" "$ORIGINAL_DATA_LIST" >> "$NEW_DATA_LIST"
grep -E "^rfc4518.spp$" "$ORIGINAL_DATA_LIST" >> "$NEW_DATA_LIST"
grep -E "^nfkc.nrm$" "$ORIGINAL_DATA_LIST" >> "$NEW_DATA_LIST"

#
# Extract desired data, and use it to build custom data files.
#

LD_LIBRARY_PATH= eval $("${INSTALL_DIR}/bin/icu-config" --invoke=icupkg) -d "$DATA_DIR" \
    -x "$NEW_DATA_LIST" "$ORIGINAL_DATA_FILE"
LD_LIBRARY_PATH= eval $("${INSTALL_DIR}/bin/icu-config" --invoke=icupkg) -s "$DATA_DIR" \
    -a "$NEW_DATA_LIST" -tl new "$ICU_DATA_FILE_LITTLE_ENDIAN_OUT"
LD_LIBRARY_PATH= eval $("${INSTALL_DIR}/bin/icu-config" --invoke=icupkg) -s "$DATA_DIR" \
    -a "$NEW_DATA_LIST" -tb new "$ICU_DATA_FILE_BIG_ENDIAN_OUT"

#
# Copy pristine sources into final destination, prune unneeded sources.
#

git clone $LIB_GIT_REPO $FRESH_SOURCE_DIR
git -C $FRESH_SOURCE_DIR checkout $LIB_GIT_REVISION
cp -r ${FRESH_SOURCE_DIR}/icu4c/* ${ICU_THIRD_PARTY_DIR}

rm -f ${ICU_THIRD_PARTY_DIR}/source/*.in             # Build system.
rm -f ${ICU_THIRD_PARTY_DIR}/source/*.m4             # Build system.
rm -f ${ICU_THIRD_PARTY_DIR}/source/install-sh       # Build system.
rm -f ${ICU_THIRD_PARTY_DIR}/source/mkinstalldirs    # Build system.
rm -f ${ICU_THIRD_PARTY_DIR}/source/runConfigureICU  # Build system.
rm -rf ${ICU_THIRD_PARTY_DIR}/as_is/                 # Scripts.
rm -rf ${ICU_THIRD_PARTY_DIR}/source/allinone/       # Workspace and project files.
rm -rf ${ICU_THIRD_PARTY_DIR}/source/config*         # Build system.
rm -rf ${ICU_THIRD_PARTY_DIR}/source/data/           # Source data.
rm -rf ${ICU_THIRD_PARTY_DIR}/source/extra/          # Non-supported API additions.
rm -rf ${ICU_THIRD_PARTY_DIR}/source/io/             # ICU I/O library.
rm -rf ${ICU_THIRD_PARTY_DIR}/source/layout/         # ICU complex text layout engine.
rm -rf ${ICU_THIRD_PARTY_DIR}/source/layoutex/       # ICU paragraph layout engine.
rm -rf ${ICU_THIRD_PARTY_DIR}/source/samples/        # Sample programs.
rm -rf ${ICU_THIRD_PARTY_DIR}/source/test/           # Test suites.
rm -rf ${ICU_THIRD_PARTY_DIR}/source/tools/          # Tools for generating the data files.
