# Generates "mongo.h" config header file containing feature flags generated by checking for the availability of certain compiler features.
# This script is invoked by the Bazel build system to generate the "mongo.h" file automatically as part of the build.
# Example usage:
# python mongo_config_header.py --compiler-path /usr/bin/gcc --compiler-args "-O2 -Wall" --output-path mongo.h
import argparse
import platform
import subprocess
import threading
import os
import json
import tempfile
from typing import Dict

logfile_path: str = ""
loglock = threading.Lock()
def log_check(message):
    global loglock, logfile_path
    with loglock:
        with open(logfile_path, "a") as f:
            f.write(message + "\n")
                
class CompilerSettings:
    compiler_path: str = ""
    compiler_args: str = ""


class HeaderDefinition:
    def __init__(self, key: str, value: str = None) -> None:
        self.key = key
        self.value = value


def compile_check(source_text: str) -> bool:
    temp = None
    if platform.system() == "Windows":
        temp = tempfile.NamedTemporaryFile(suffix=".cpp", delete=False)
        temp.write(source_text.encode())
        temp.close()
        command = [
            CompilerSettings.compiler_path,
            "/c",  # only compile and assemble, don't link since we don't want to have to pass in all of the libs of the dependencies
            temp.name,
            *CompilerSettings.compiler_args.split(" "),
        ]
        log_check(" ".join(command[:-1] + [source_text]))
        result = subprocess.run(
            command,
            capture_output=True,
            text=True,
            env={**os.environ.copy(), **CompilerSettings.env_vars},
        )
    else:
        command = [
            CompilerSettings.compiler_path,
            "-c",  # only compile and assemble, don't link since we don't want to have to pass in all of the libs of the dependencies
            "-x",
            "c++",
            *CompilerSettings.compiler_args.split(" "),
            "-",
        ]
        log_check(" ".join(command + [source_text]))
        result = subprocess.run(
            command,
            input=source_text,
            capture_output=True,
            text=True,
            env={**os.environ.copy(), **CompilerSettings.env_vars},
        )
    if result.returncode != 0:
        log_check(f"stdout:\n{result.stdout}")
        log_check(f"stderr:\n{result.stderr}")
    log_check(f"Exit code:\n{result.returncode}")
    log_check("--------------------------------------------------\n\n")
    if temp:
        os.unlink(temp.name)
    return result.returncode == 0


def func_check(function_name: str, header_path: str = "<string.h>") -> bool:
    source_text = """
#include <assert.h>
#include %(header_path)s

int main(void) {
  return 0 ? %(function_name)s != nullptr : 1;
}
""" % {"header_path": header_path, "function_name": function_name}
    return compile_check(source_text)

def header_check(header: str) -> bool:
    source_text = """
        #include %(header)s
        int main(int argc, char* argv[]) {
            return 0;
        }
        """ % {"header": header}
    return compile_check(source_text)

def fallocate_flag() -> list[HeaderDefinition]:
    log_check("[HAVE_FALLOCATE] Checking for fallocate...")

    # Glibc 2.25+, OpenBSD 5.5+ and FreeBSD 11.0+ offer explicit_bzero, a secure way to zero memory
    if func_check("fallocate", "<fcntl.h>"):
        return [HeaderDefinition("HAVE_FALLOCATE")]
    else:
        return []

def sync_file_range_flag() -> list[HeaderDefinition]:
    log_check("[HAVE_SYNC_FILE_RANGE] Checking for sync_file_range...")

    # Glibc 2.25+, OpenBSD 5.5+ and FreeBSD 11.0+ offer explicit_bzero, a secure way to zero memory
    if func_check("sync_file_range", "<fcntl.h>"):
        return [HeaderDefinition("HAVE_SYNC_FILE_RANGE")]
    else:
        return []

def x86intrin_header_present() -> list[HeaderDefinition]:
    log_check("[HAVE_X86INTRIN_H] Checking for x86intrin.h...")
    
    if platform.machine() == "x86_64":
        if header_check("<x86intrin.h>"):
            return [HeaderDefinition("HAVE_X86INTRIN_H")]

    return []

def arm_neon_header_present() -> list[HeaderDefinition]:
    log_check("[HAVE_ARM_NEON_INTRIN_H] Checking for arm_neon.h...")
    if header_check("<arm_neon.h>"):
        return [HeaderDefinition("HAVE_ARM_NEON_INTRIN_H")]
    
    return []




def generate_config_header(compiler_path, compiler_args, env_vars, logpath, additional_inputs=[], extra_definitions={}) -> Dict[str, str]:
    global logfile_path
    CompilerSettings.compiler_path = compiler_path
    CompilerSettings.compiler_args = compiler_args
    CompilerSettings.env_vars = {
        **json.loads(env_vars),
    }
    if platform.system() == "Linux":
        CompilerSettings.compiler_args += " -D_GNU_SOURCE"

    logfile_path = logpath

    definitions: list[HeaderDefinition] = []

    definitions += arm_neon_header_present()
    definitions += x86intrin_header_present()
    definitions += sync_file_range_flag()
    definitions += fallocate_flag()

    subst_keys = {
        'HAVE_FALLOCATE',
        'HAVE_SYNC_FILE_RANGE',
        'HAVE_X86INTRIN_H',
        'HAVE_ARM_NEON_INTRIN_H',
    }

    define_map = { definition.key: definition.value or "" for definition in definitions}
    subst_map = {}
    for subst, define in [(f"@{key}@", key) for key in subst_keys]:
        if define not in define_map:
            log_check(f"{define} was unchecked and is unused.")
            subst_map[subst] = f"// # undef {define}"
        else:
            subst_map[subst] = f"#define {define} {define_map[define]}"
    return subst_map
